//
//  KeyDriverProtocol.h
//  KeyDriverProtocol
//
//  v1.3

#import <UIKit/UIKit.h>


@protocol KeyDriverProtocol


@required

/*!
 @function
 @abstract   Check a Device Connection Status
 @param		[out]	pConnected  		If device is connected or not
 @result     Error code 
 */

-(long )isConnected:(bool *)pConnected;

/*!
 @function
 @abstract   Generate a Key-pair
 @param		[in]	pPassword			Device password
 [in]	publicKeyLength		Length of public key (bit), 1024 or 2048 
 [out]	ppPublicKey			Public key of the generated key-pair (format comply with the coding specification described in this document)
 @result     Error code 
 */
-(long ) createRSAKey:(NSString *)pPassword publicKeyLength:( NSInteger)publicKeyLength
											ppPublicKey:(NSData **)ppPublicKey; 

/*!
 @function
 @abstract   Import a Public Key Certificate
 @param		[in]	pCertificate		Der coded X509 public key certificate 
 @result     Error code 
 */
-(long )importX509Certificate:(NSData *)pCertificate;


/*!
 @function
 @abstract   Decrypt Digital Envelope
 @param		[in]	pCertificate 		Public key certificate corresponding to the private key, which is the decrypting certificate
 [in]	pPassword			Device password
 [in]	inData				Cipher text of digital envelope 
 [out]	ppOutData			Original text of digital envelope
 @result     Error code 
 */
-(long )decryptEnvelopeData: (NSData *) pCertificate
				   pPassword:(NSString *) pPassword
					  inData:( NSData *) inData
				   ppOutData:(NSData**) ppOutData;


/*!
 @function
 @abstract   Digital Sign (including explicit key signature, ordinary signature, SSL handling, and P10 signature)
 @param		[in]	pCertificate		Public key certificate corresponding to private key, and if RSA key pairs are just generated, use the public key value returned from createRSAkey function. 
 [in]	pPassword			Device password
 [in]	srcData				Source data to be signed
 [in]	pHashOID			Hash algorithm identification
 [out]	ppOutData			Signed data(PKCS1 format，without adding complementing bits)
 @result     Error code 
 */
-(long )signData:(NSData *)pCertificate
	   pPassword:(NSString *)pPassword
		 srcData:(NSData *)srcData
		pHashOID:(NSString *)pHashOID
	   ppOutData:(NSData **)ppOutData;

/*!
 @function
 @abstract   Change a Device Password
 @param		[in]    pCertificate		Public key certificate, and if it’s device password, ignore this parmeters.
 [in]	oldPwd				Old password
 [in]	newPwd				New password
 @result     Error code 
 */
-(long ) changePassword:(NSData *)pCertificate  oldPwd:(NSString *)oldPwd newPwd:(NSString *)newPwd;

/*!
 @function
 @abstract   Verity password
 @param		
 [in]		password				password
 @result     Error code 
 */
-(long)verifyPassword:(NSString *)password;

/*!
 @function
 @abstract   Reset password
 @param		
 [in]		password				password
 @result     Error code 
 */
-(long)resetPassword:(NSString *)password;

/*!
 @function
 @abstract   Get DER Code of Public Key Certificate
 @param		[out]	ppCertificates		Public key certificate DER coded NSData array
 @result     Error code	
 */
-(long ) getX509Certificates:(NSArray **)ppCertificates;

/*!
 @function
 @abstract   Get the Remaining Number of Password Attempts
 @param		[in]	pCertificate		Public key certificate, and if it’s the device password, ignore this parameter.
 [out]	pRemaining			The remaining number of password attempts
 @result     Error code 
 */
-(long ) getPwdCanRetries:(NSData *)pCertificate pRemaining:(int *)pRemaining;

/*!
 @function
 @abstract   Get a Device ID (the unique ID of a device)
 @param      [out]	ppDeviceSN			Device ID
 @result     Error code
 */
-(long ) getDeviceSerialNumber:(NSString **)ppDeviceSN;


/*!
 @function
 @abstract   Get random number
 @param		[in]	random_len    Random number length	
            [out]	randomNumber  Random
 @result    Error code 
 */
-(long)getRandomNumber:(NSString **)randomNum ran_len:(unsigned long)ran_len;



@end
